CLOSE DATABASES ALL
SET SAFETY OFF
? TIME(), REPLICATE("=", 50)
loTex = CREATEOBJECT("FoxToTex")
loTable = CREATEOBJECT("CursorToTable")

CREATE CURSOR c_Biblio (cBibEntry C(100))
INSERT INTO c_Biblio VALUES ("One " + TIME())
INSERT INTO c_Biblio VALUES ("Two " + TIME())
INSERT INTO c_Biblio VALUES ("Three " + TIME())

CREATE CURSOR c_Description (cTitle C(100), cEntry C(250), nNumber N)
tcItem = "Purpose"
INSERT INTO c_Description VALUES (tcItem + loTex.TexIndexItem(tcItem), "This example is to show what description lists are like.", RAND() * 10000)
tcItem = "Example"
INSERT INTO c_Description VALUES (tcItem + loTex.TexIndexItem(tcItem), "Almost anything can go in here & often does.", RAND() * 10000)
tcItem = "Element"
INSERT INTO c_Description VALUES (tcItem + loTex.TexIndexItem(tcItem), "My Description of " + tcItem + ".", RAND() * 10000)


IF VARTYPE(loTex) = "O"
	WITH loTex
		.cOutputFile		= [texOutput_] + DTOC(DATE(),1) + [.tex]
		.cDocumentClass 	= [book]
		.cDocumentName 		= [Kevin's Book]
		
		.cOptionFontSize 	= [12pt]
		.cOptionPaperSize 	= [letterpaper]
		.cOptionPageFormat 	= [onecolumn]
		
		.cTitle 			= [Kevin's Book & Story \index{Kevin}]
		.cAuthor 			= [Kevin Cully]
		.dDate 				= DATE()
		
		.cBiblioCursor		= [c_Biblio]
		.cBiblioField		= [cBibEntry]
		
		.cDocument = []	
		.cDocument = .cDocument + .TexFrontMatter()
		.cDocument = .cDocument + .TexListDescription("c_Description", "cTitle", "cEntry")
		
		IF VARTYPE(loTable) = "O"
			loTable.cSourceCursor = [c_Description]
			loTable.cTabularBanner = [Test Table Banner ] + DTOC(DATE(),1)
			&& loTable.cTableCaption = [First Table]
			.cDocument = .cDocument + loTable.GenerateLongTable()
			.cDocument = .cDocument + .TexPageBreak(3)
			
			.cDocument = .cDocument + .TexParagraph("Another table", "We're going to print the same table again.  It goes right below this paragraph.")
			&& loTable.cTableCaption = [Second Table]
			.cDocument = .cDocument + loTable.GenerateLongTable()	&& Making two tables in the document
		ENDIF
		
		.cDocument = .cDocument + .TexBackMatter()
		.cDocument = .TexInsertPackages(.cDocument)
		.TexOutput()
		.TeXToPDF()
	ENDWITH
ELSE
	? TIME(), [Could not instantiate the object FoxToTex"]
ENDIF
? TIME(), REPLICATE("-", 50)


*!************************************************
*!************************************************
*!************************************************
*!*	Many thanks to the following people and organizations:
*!*		Norm Walsh (http://nwalsh.com/tex/)
*!*		... and many other TeX and LaTeX sites on the internet.
*!* PURPOSE:
*!*		This class' purpose is to help VFP developers get the most out of
*!*	TeX.  Many of the methods are meant to just give reference and examples
*!*	on how to accomplish a task in TeX.  
*!*	
*!* Note that when using \makeindex, that you may need to make multiple 
*!*	passes to generate the full document.  First call PDFLATEX, then
*!*	MAKEINDEX {just stem of file}, and then call PDFLATEX again to 
*!* have all of the index entries included in the resulting file.
DEFINE CLASS FoxToTex AS Custom

	CRLF				= []
	cOutputFile			= [texFile.tex]		&& Output file
	cDocument			= []				&& Actually holds the output
	
	
	cDocumentClass 		= []				&& article, book, report, letter
	cDocumentName		= []

	lShowTableOfContents= .T.				&& Determines whether a TOC is created.
	lShowIndex			= .T.				&& Determines whether an index is created.
	cPageStyle			= []				&& Headings
	
	cOptionFontSize		= []				&& 10pt, 11pt, 12pt
	cOptionPaperSize 	= []				&& letterpaper, legalpaper, executivepaper
	cOptionPageFormat	= []				&& onecolumn, twocolumn, etc.

	cUsePackageOptions	= []				&& Screen,rightpanel,paneltox,code
	cUsePackage			= [varioref]		&& pdfscreen, varioref
	
	&& Front Matter settings
	cTitle				= []				&& Title Text
	cAuthor				= []				&& Author names and addresses
	dDate				= []				&& Date Text

	&& Abstract settings: In document class REPORT, the abstract appears on a separate page without
	&& a page number; in ARTICLE, it comes after the title heading on the first page. An abstract
	&& is not possible in document class BOOK. - From Chapter 3 tutorial on Tex.
	cAbstract			= []
	
	cBiblioCursor		= []
	cBiblioField		= []

*!************************************************
*!* Program: Init
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:01:16 AM
*!* Copyright:
*!* Description: Takes care of setting up our object for use.
*!* Revision Information:	
FUNCTION Init
	THIS.CRLF = CHR(10)
	
	WAIT WINDOW THIS.Version() NOWAIT
	
	*** We use this list for when calling methods that require a package such as THIS.IncludeGraphics().
	*** Populated through THIS.AddPackage(tcPackage, tcOptions)
	IF USED("c_UsePackage")
		USE IN c_UsePackage
	ENDIF
	CREATE CURSOR c_UsePackage(cPackage C(200), cOptions C(200))

	WAIT CLEAR
		
	DODEFAULT()
ENDFUNC

*!************************************************
*!* Program: OutputFileNotInUse
*!* Author: CULLY Technologies, LLC
*!* Date: 07/19/06 09:42:29 AM
*!* Copyright:
*!* Description: Checks that the output file is not in
*!*		use.  Returns a boolean .T. if it is NOT in
*!*	use.
*!* Revision Information:
FUNCTION OutputFileNotInUse(tcOutputFile AS String) AS Boolean
     LOCAL llRetVal, lcOutputFile, lnFileHandle, lcPDFFile, lnAttribute
     llRetVal = .T.
     IF PCOUNT() = 1 AND NOT EMPTY(tcOutputFile)
     	lcOutputFile = tcOutputFile
     ELSE
     	lcOutputFile = THIS.cOutputFile
     ENDIF
     IF NOT EMPTY(lcOutputFile)
     	lcPDFFile = FORCEEXT(lcOutputFile, 'PDF')
	     IF FILE(lcPDFFile)
	     	lnAttribute = 12	&& Read and Write, Unbuffered
			lnFileHandle = FOPEN(lcPDFFile, lnAttribute)
			IF lnFileHandle < 1
				llRetVal = .F.		&& The file is in use by another process.  Probably the Adobe Acrobat Reader.
			ENDIF
			=FCLOSE(lnFileHandle)
	     ENDIF
     ENDIF
    RETURN llRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexRender
	THIS.cDocument = []
	THIS.cDocument = THIS.cDocument + THIS.TexFrontMatter()
	THIS.cDocument = THIS.cDocument + THIS.TexMiddleMatter()
	THIS.cDocument = THIS.cDocument + THIS.TexBackMatter()
	? TIME(), [>>>>]
	? THIS.cDocument
	? TIME(), [<<<<]
RETURN THIS.cDocument
ENDFUNC

*!************************************************
*!* Program: TexOutput
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 11:41:32 AM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexOutput() AS Integer
	LOCAL lnRetVal, lcOutputFile
	lcOutputFile = FULLPATH(THIS.cOutputFile)
	&& lnRetVal = STRTOFILE(THIS.cDocument, lcOutputFile)
	lnRetVal = STRTOFILE(THIS.cDocument, THIS.cOutputFile)
	? TIME(), THIS.cOutputFile, lnRetVal
	RETURN lnRetVal
ENDFUNC

*!************************************************
*!* Program: TeXToPDF
*!* Author: CULLY Technologies, LLC
*!* Date: 07/15/06 05:30:22 PM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TeXToPDF()
     LOCAL llRetVal
     llRetVal = .T.
     lcCommand = [PDFLatex ] + THIS.cOutputFile
     RUN &lcCommand
     
    RETURN llRetVal
ENDFUNC

*!************************************************
*!* Program: TeXOpenPDF
*!* Author: CULLY Technologies, LLC
*!* Date: 07/15/06 05:49:04 PM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TeXOpenPDF(tcFileName AS String) AS Boolean
     LOCAL llRetVal, lcCommand
     llRetVal = .F.
     IF JUSTEXT(tcFileName) = "PDF"
	     IF FILE(tcFileName)
	     	lcCommand = [ACROBAT ] + tcFileName
	     	RUN /N &lcCommand
	     	llRetVal = .T.
	     ENDIF
     ENDIF
    RETURN llRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexFrontMatter() AS String
	LOCAL lcRetVal
	lcRetVal = []
	lcRetVal = lcRetVal + THIS.TexDocumentClass()
	
	lcRetVal = lcRetVal + THIS.TeXPackagePlaceholder()
	
	lcRetVal = lcRetVal + [\title{] + THIS.TexEscChars( THIS.cTitle ) + [}] + THIS.CRLF
	lcRetVal = lcRetVal + [\author{] + THIS.TexEscChars( THIS.cAuthor ) + [}] + THIS.CRLF
	lcRetVal = lcRetVal + [\date{] + TRANSFORM(THIS.dDate) + [}] + THIS.CRLF
	lcRetVal = lcRetVal + THIS.TexIndexMake() + THIS.CRLF

	lcRetVal = lcRetVal + THIS.TexBegin()
	lcRetVal = lcRetVal + [\maketitle] + THIS.CRLF + THIS.CRLF
	
	lcRetVal = lcRetVal + THIS.TexAbstract()
	lcRetVal = lcRetVal + THIS.TexTableOfContents()
RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexDocumentClass()
	LOCAL lcRetVal
	lcRetVal = [\documentclass] + THIS.TexOptions() + [{] + THIS.cDocumentClass + [}] + THIS.CRLF
RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexOptions()
	LOCAL lcRetVal
	lcRetVal = []
	IF NOT EMPTY(THIS.cOptionFontSize+THIS.cOptionPaperSize+THIS.cOptionPageFormat)
		lcRetVal = lcRetVal + IIF(EMPTY(lcRetVal), [], [,]) + IIF(EMPTY(THIS.cOptionFontSize), [], THIS.cOptionFontSize)
		lcRetVal = lcRetVal + IIF(EMPTY(lcRetVal), [], [,]) + IIF(EMPTY(THIS.cOptionPaperSize), [], THIS.cOptionPaperSize)
		lcRetVal = lcRetVal + IIF(EMPTY(lcRetVal), [], [,]) + IIF(EMPTY(THIS.cOptionPageFormat), [], THIS.cOptionPageFormat)
		lcRetVal = "[" + lcRetVal
		lcRetVal = lcRetVal + "]"
	ENDIF
RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TeXPackagePlaceholder
*!* Author: CULLY Technologies, LLC
*!* Date: 07/19/06 10:59:47 AM
*!* Copyright:
*!* Description: Generates the text for the package placeholder
*!* Revision Information:
FUNCTION TeXPackagePlaceholder
    LOCAL lcRetVal
    lcRetVal = THIS.TeXComment("<<Placeholder for packages>>") + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TeXInsertPackages
*!* Author: CULLY Technologies, LLC
*!* Date: 07/15/06 04:56:32 PM
*!* Copyright:
*!* Description: This method, takes all of the packages needed by the preceeding calls, 
*!* 	and inserts them into the placeholder.
*!* Revision Information:
*!*		7/19/2006 KJC - Changed reference to the package placeholder by calling the method instead of hard coding.
FUNCTION TeXInsertPackages(tcTeXDocument AS String) AS String
    LOCAL lcRetVal, lcPackageList, lcPPH
    lcRetVal = []
    IF USED("c_UsePackage")
    	lcPackageList = []
    	SELECT c_UsePackage
    	SCAN
    		SCATTER NAME loP
    		lcPackageList = lcPackageList + THIS.TeXUsePackage(ALLTRIM(loP.cPackage), ALLTRIM(loP.cOptions))
    	ENDSCAN
    	IF EMPTY(lcPackageList)
    		lcPackageList = THIS.TexComment("No packages were needed.") + THIS.CRLF
    	ENDIF
    	lcPackageList = lcPackageList + THIS.CRLF		&& The extra line here helps with the spacing and makes the output more readible.
		lcPPH = THIS.TeXPackagePlaceholder()
		
    	lcRetVal = STRTRAN(tcTeXDocument, lcPPH, lcPackageList)
    ENDIF
    RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexUsePackage(tcPackage AS String, tcPackageOptions AS String) AS String
	LOCAL lcRetVal
	lcRetVal = []
	IF NOT EMPTY(tcPackage) OR NOT EMPTY(tcPackageOptions)
		lcRetVal = [\usepackage]
		lcRetVal = lcRetVal + IIF(EMPTY(tcPackageOptions), [], "[" + tcPackageOptions + "]")
		lcRetVal = lcRetVal + IIF(EMPTY(tcPackage), [], "{" + tcPackage + "}")
		lcRetVal = lcRetVal + THIS.CRLF
	ENDIF
RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexAddPackage
*!* Author: CULLY Technologies, LLC
*!* Date: 07/15/06 04:40:07 PM
*!* Copyright:
*!* Description: Adds a package to the cursor c_UsePackage.  On the creation of the TeX file,
*!*		the cursor is used to include all necessary packages.
*!* Revision Information:
FUNCTION TeXAddPackage(tcPackage AS String, tcOptions AS String) AS Boolean
    LOCAL llRetVal
    llRetVal = .F.
    IF USED("c_UsePackage")
    	tcPackage = NVL(tcPackage, [])
    	IF PCOUNT() = 1
	   		RELEASE tcOptions
    		tcOptions = []
    	ELSE
	    	tcOptions = NVL(tcOptions, [])
    	ENDIF
    	IF NOT EMPTY(tcPackage)
    		lnOrigArea = SELECT()
    		SELECT c_UsePackage
    		LOCATE FOR ALLTRIM(UPPER(cPackage)) = ALLTRIM(UPPER(tcPackage))
    		IF FOUND()
    		ELSE
    			INSERT INTO c_UsePackage (cPackage, cOptions) VALUES (tcPackage, tcOptions)
    			llRetVal = .T.
    		ENDIF
    		SELECT (lnOrigArea)
    	ENDIF
    ENDIF
    RETURN llRetVal
ENDFUNC

*!************************************************
*!* Program: TeXPageStyle
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 07:27:23 PM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexPageStyle()
	LOCAL lcRetVal
	lcRetVal = IIF(EMPTY(THIS.cPageStyle), [], [\pagestyle{] + THIS.cPageStyle + [}] + THIS.CRLF)
RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextColor
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 07:23:45 PM
*!* Copyright:
*!* Description: Takes the text and will change it's color from the default.
*!*	Example: \textcolor{red}{text in red}
*!* Revision Information:
FUNCTION TeXTextColor(tcColor AS String, tcText AS String) AS String
    LOCAL lcRetVal
    lcRetVal = [\textcolor{] + tcColor + [}{] + ALLTRIM(tcText) + [}]
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextBold
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 07:23:45 PM
*!* Copyright:
*!* Description: Takes the text and will change it to bold.
*!*	Example: \textbf{This text is now bold!}
*!* Revision Information:
FUNCTION TeXTextBold(tcText AS String) AS String
    LOCAL lcRetVal
    lcRetVal = [\textbf{] + ALLTRIM(tcText) + [}]
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextItalics
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 07:23:45 PM
*!* Copyright:
*!* Description: Takes the text and will change it to italics.
*!*	Example: \textbf{This text is now in italics!}
*!* Revision Information:
FUNCTION TeXTextItalics(tcText AS String) AS String
    LOCAL lcRetVal
    lcRetVal = [\textit{] + ALLTRIM(tcText) + [}]
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextSlanted
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 07:23:45 PM
*!* Copyright:
*!* Description: Takes the text and will change it to slanted font.
*!*	Example: \textsl{This text is now slanted!}
*!* Revision Information:
FUNCTION TeXTextSlanted(tcText AS String) AS String
    LOCAL lcRetVal
    lcRetVal = [\textsl{] + ALLTRIM(tcText) + [}]
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextSmallCaps
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 07:23:45 PM
*!* Copyright:
*!* Description: Takes the text and will change it to italics.
*!*	Example: \textsc{This text is now SMALL CAPS!}
*!* Revision Information:
FUNCTION TeXTextSmallCaps(tcText AS String) AS String
    LOCAL lcRetVal
    lcRetVal = [\textsc{] + ALLTRIM(tcText) + [}]
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextMonospace
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 07:23:45 PM
*!* Copyright:
*!* Description: Takes the text and will change it to a monospaced font.
*!*	Example: \texttt{This text is now MONOSPACED!}
*!* Revision Information:
FUNCTION TeXTextMonospace(tcText AS String) AS String
    LOCAL lcRetVal
    lcRetVal = [\texttt{] + ALLTRIM(tcText) + [}]
    RETURN lcRetVal
ENDFUNC


*!************************************************
*!* Program: TexComment
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 04:51:08 PM
*!* Copyright:
*!* Description: Creates a TeX comment tag
*!* Revision Information:
FUNCTION TexComment(tcComment AS String) AS String
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = [% ] + tcComment + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexBegin
*!* Author: CULLY Technologies, LLC
*!* Date: 06/19/06 04:53:11 PM
*!* Copyright:
*!* Description: Marks the beginning of the TeX document.
*!* Revision Information:
FUNCTION TexBegin()
	RETURN [\begin{document}] + THIS.CRLF
ENDFUNC

*!************************************************
*!* Program: TexBeginCenter
*!* Author: CULLY Technologies, LLC
*!* Date: 07/16/06 11:01:52 PM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexBeginCenter
    LOCAL lcRetVal
    lcRetVal = [\begin{center}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexEndCenter
*!* Author: CULLY Technologies, LLC
*!* Date: 07/16/06 11:01:52 PM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexEndCenter
    LOCAL lcRetVal
    lcRetVal = [\end{center}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexAbstract
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 08:41:19 AM
*!* Copyright:
*!* Description: Creates the section that holds the abstract for the article, etc.
*!* Revision Information:
*!*		5/13/06 KJC - Books do not have abstracts
FUNCTION TexAbstract()
	LOCAL lcRetVal
	lcRetVal = []
	IF NOT EMPTY(THIS.cAbstract)
		IF ALLTRIM(UPPER(THIS.cDocumentClass)) # [BOOK]
			lcRetVal = lcRetVal + [\begin{abstract}] + THIS.CRLF
			lcRetVal = lcRetVal + THIS.cAbstract
			lcRetVal = lcRetVal + [\end{abstract}] + THIS.CRLF + THIS.CRLF
		ENDIF
	ENDIF
RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexMakeTitle
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 10:50:00 AM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexMakeTitle
	LOCAL lcRetVal
	lcRetVal = [\maketitle] + THIS.CRLF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TableOfContents
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 10:51:12 AM
*!* Copyright:
*!* Description: Creates the table of contents tag
*!* Revision Information:
FUNCTION TexTableOfContents
	LOCAL lcRetVal
	IF THIS.lShowTableOfContents
		lcRetVal = [\tableofcontents] + THIS.CRLF
	ENDIF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexMiddleMatter
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:15:53 AM
*!* Copyright:
*!* Description: Takes care of calling the middle matter section
*!* Revision Information:
FUNCTION TexMiddleMatter AS String
	LOCAL lcRetVal
	lcRetVal = []
RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexChapter(tcChapterTitle AS String) AS String
	LOCAL lcRetVal
	lcRetVal = []
	IF VARTYPE(tcChapterTitle) = "C"
		lcRetVal = [\chapter{] + NVL(tcChapterTitle,[]) + [}] + THIS.CRLF
	ENDIF
RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexSection(tcSectionTitle AS String, tcSectionText AS String) AS String
	LOCAL lcRetVal
	lcRetVal = []
	IF PCOUNT() = 2
		lcRetVal = [\section{] + NVL(tcSectionTitle,[]) + [}] + THIS.CRLF
		lcRetVal = lcRetVal + NVL(tcSectionText,[]) + THIS.CRLF + THIS.CRLF
	ENDIF
RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexSubSection(tcSectionTitle AS String, tcSectionText AS String) AS String
	LOCAL lcRetVal
	lcRetVal = THIS.TexSection(tcSectionTitle, tcSectionText)
	lcRetVal = STRTRAN(lcRetVal, [\section], [\subsection])
RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexSubSubSection(tcSectionTitle AS String, tcSectionText AS String) AS String
	LOCAL lcRetVal
	lcRetVal = THIS.TexSection(tcSectionTitle, tcSectionText)
	lcRetVal = STRTRAN(lcRetVal, [\section], [\subsubsection])
RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexParagraph(tcParagraphTitle AS String, tcParagraphText AS String) AS String
	LOCAL lcRetVal
	lcRetVal = []
	IF PCOUNT() = 2
		lcRetVal = [\paragraph{] + NVL(tcParagraphTitle,[]) + [}] + THIS.CRLF
		lcRetVal = lcRetVal + NVL(tcParagraphText,[]) + THIS.CRLF + THIS.CRLF
	ENDIF
RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TeXNoIndent
*!* Author: CULLY Technologies, LLC
*!* Date: 07/18/06 08:11:54 PM
*!* Copyright:
*!* Description: When used at the beginning of the paragraph, it suppresses the paragraph indentation. It has no effect when used in the middle of a paragraph.
*!* Revision Information:
FUNCTION TeXNoIndent
    LOCAL lcRetVal
    lcRetVal = [\noindent ]
    RETURN lcRetVal
ENDFUNC
*!************************************************
*!* Program: TexClearPage
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 09:16:05 AM
*!* Copyright:
*!* Description: The \clearpage command ends the current page and causes all figures 
*!*		and tables that have so far appeared in the input to be printed.
*!* Revision Information:
FUNCTION TexClearPage
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\clearpage] + THIS.CRLF()
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexPageBreak
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 09:02:42 AM
*!* Copyright:
*!* Description: The \pagebreak command tells LaTeX to break the current page at the 
*!*		point of the command. With the optional argument, number, you can convert the 
*!*		\pagebreak command from a demand to a request. The number must be a number 
*!*		from 0 to 4. The higher the number, the more insistent the request is.
*!*		A value of 4 is the same as passing no parameter at all which is to say that
*!*		it's a demand as opposed to a request.
*!*		
*!*	Programming Note:
*!*		I've broken the conditional statements into multiple statements as a debugging
*!*		tool.  Yes, I know it will slow down the execution but I think it's worth it
*!*		to be a bit clearer on where it is kicking out.
*!* Revision Information:
FUNCTION TexPageBreak(tnInsistance AS Integer) AS String
    LOCAL lcRetVal
    lcRetVal = [\pagebreak]
    IF PCOUNT() = 1
    	IF NOT ISNULL(tnInsistance) 
    		IF VARTYPE(tnInsistance) = "N"
    			IF BETWEEN(tnInsistance, 0, 4)
    				lcRetVal = lcRetVal + "[" + TRANSFORM(INT(tnInsistance)) + "]"
    			ENDIF
    		ENDIF
    	ENDIF
    ENDIF
    lcRetVal = lcRetVal + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexList
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:21:20 AM
*!* Copyright:
*!* Description: A generic method that takes care of creating a TexList.
*!*		Called from other list methods.
*!* Revision Information:
FUNCTION TexList(tcSourceCursor AS String, tcField AS String, tcTag AS String) AS String
	LOCAL lcRetVal, lnOrigArea, loList, lcItem
	lcRetVal = []
	IF PCOUNT() = 3
		IF NOT EMPTY(NVL(tcTag,[]))
			IF USED(tcSourceCursor)
				lnOrigArea = SELECT()
				SELECT (tcSourcCursor)
				IF NOT EMPTY(FIELD(tcField))
					lcRetVal = lcRetVal + [\begin{] + tcTag + [}] + THIS.CRLF
					SCAN
						SCATTER NAME loList
						lcItem = EVALUATE([loList.] + tcField)
						lcRetVal = lcRetVal + [\item ] + NVL(lcItem,[]) + THIS.CRLF
					ENDSCAN
					lcRetVal = lcRetVal + [\end{] + tcTag + [}] + THIS.CRLF
				ENDIF
				SELECT (lnOrigArea)
			ENDIF
		ENDIF
	ENDIF
RETURN lcRetVal

*!************************************************
*!* Program: TexListEnumerate
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:28:06 AM
*!* Copyright:
*!* Description: This returns a list of type enumerate.
*!*		Enumerated lists are numbered lists.  Eg: (1) something (2) something else
*!* Revision Information:
FUNCTION TexListEnumerate(tcSourceCursor AS String, tcField AS String) AS String
	LOCAL lcRetVal
	RETURN THIS.TexList(tcSourceCursor, tcField, [enumerate])
ENDFUNC

*!************************************************
*!* Program: TexListItemize
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:28:06 AM
*!* Copyright:
*!* Description: This returns a list of type ITEMIZE.
*!*		Itemized lists are bulleted lists. Eg: * Something * something else
*!* Revision Information:
FUNCTION TexListItemize(tcSourceCursor AS String, tcField AS String) AS String
	LOCAL lcRetVal
	RETURN THIS.TexList(tcSourceCursor, tcField, [itemize])
ENDFUNC

*!************************************************
*!* Program: TexListDescription
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:28:06 AM
*!* Copyright:
*!* Description: This returns a list of type DESCRIPTION.
*!*		Description lists are lists. Eg: * Something * something else
*!* Revision Information:
FUNCTION TexListDescription(tcSourceCursor AS String, tcTitleField AS String, tcDescField AS String) AS String
	LOCAL lcRetVal, lnOrigArea, loList, lcItemTitle, lcItemDesc
	lcRetVal = []
	IF PCOUNT() = 3
		IF NOT EMPTY(NVL(tcTitleField,[])) AND NOT EMPTY(NVL(tcDescField,[]))
			IF USED(tcSourceCursor)
				lnOrigArea = SELECT()
				SELECT (tcSourceCursor)
				IF NOT EMPTY(FIELD(tcTitleField)) AND NOT EMPTY(FIELD(tcDescField))
					lcRetVal = lcRetVal + [\begin{description}] + THIS.CRLF
					SCAN
						SCATTER NAME loList
						lcItemTitle = ALLTRIM( EVALUATE([loList.] + tcTitleField) )
						lcItemDesc = ALLTRIM( EVALUATE([loList.] + tcDescField) )
						
						lcItemTitle = THIS.TexEscChars( lcItemTitle )
						lcItemDesc = THIS.TexEscChars ( lcItemDesc )
						
						lcRetVal = lcRetVal + "\item[" + NVL(lcItemTitle,[]) + "]"
						lcRetVal = lcRetVal + NVL(lcItemDesc,[]) + THIS.CRLF
					ENDSCAN
					lcRetVal = lcRetVal + [\end{description}] + THIS.CRLF
				ENDIF
				SELECT (lnOrigArea)
			ENDIF
		ENDIF
	ENDIF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexText
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:45:50 AM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexText(tcEntry AS String, tcTag AS String) AS String
	LOCAL lcRetVal, lcEntry, lcTag
	lcRetVal = []
	lcEntry = NVL(tcEntry,[])
	lcTag = NVL(tcTag,[])
	IF NOT EMPTY(lcEntry) AND NOT EMPTY(lcTag)
		lcRetVal = lcRetVal + [\begin{] + lcTag + [}] + THIS.CRLF
		lcRetVal = lcRetVal + lcEntry + THIS.CRLF
		lcRetVal = lcRetVal + [\end{] + lcTag + [}] + THIS.CRLF + THIS.CRLF
	ENDIF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextQuote
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:50:40 AM
*!* Copyright:
*!* Description: Takes care of calling to the sub method of TexText()
*!*		to create a QUOTE.  A quote is an indented block of text
*!*		that is a single paragraph.
*!* Revision Information:
FUNCTION TexTextQuote(tcEntry AS String) AS String
	LOCAL lcRetVal
	lcRetVal = THIS.TexText(tcEntry, [quote])
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextQuotation
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:50:40 AM
*!* Copyright:
*!* Description: Takes care of calling to the sub method of TexText()
*!*		to create a QUOTATION. Quotations are blocks of text that can
*!*		span many paragraphs.
*!* Revision Information:
FUNCTION TexTextQuotation(tcEntry AS String) AS String
	LOCAL lcRetVal
	lcRetVal = THIS.TexText(tcEntry, [quotation])
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexTextVerse
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 09:50:40 AM
*!* Copyright:
*!* Description: Takes care of calling to the sub method of TexText()
*!*		to create a VERSE.  Poetry is displayed with the VERSE env.
*!* Revision Information:
FUNCTION TexTextVerse(tcEntry AS String) AS String
	LOCAL lcRetVal
	lcRetVal = THIS.TexText(tcEntry, [verse])
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TeXFigure
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 10:04:45 AM
*!* Copyright:
*!* Description: Places a graphic into the TeX and labels it as a figure.  Calls TeXIncludeGraphics.
*!*		The scale parameter can be "width=1in" or "height=1.5in"
*!*		or "scale=.25", and combinations similar.
*!*	Example: THIS.TexFigure([berberis3.jpg], [width=2in], [Barberry], [Lots of Thorns!])

*!* Revision Information:
FUNCTION TeXFigure(tcGraphic AS String, tcScale AS String, tcCaption AS String, tcLabel AS String) AS String
	LOCAL lcRetVal, lcGraphic, tcCaption, tcFigureRef, lcScale
	lcRetVal = []
	lcGraphic = NVL(tcGraphic, [])
	lcScale = NVL(tcScale, [])
	lcCaption = NVL(tcCaption, [])
	lcLabel = NVL(tcLabel, [])
	IF FILE(tcGraphic)
		THIS.TeXAddPackage("graphicx")
		lcRetVal = lcRetVal + [\begin{figure}] + THIS.CRLF
		lcRetVal = lcRetVal + [\centering] + THIS.CRLF
		lcRetVal = lcRetVal + THIS.TexIncludeGraphics(tcGraphic, tcScale)
		lcRetVal = lcRetVal + IIF(EMPTY(lcCaption),[], [\caption{]+lcCaption+[}] + THIS.CRLF)
		lcRetVal = lcRetVal + IIF(EMPTY(lcLabel),[], [\label{]+lcLabel+[}] + THIS.CRLF)
		lcRetVal = lcRetVal + [\end{figure}] + THIS.CRLF
	ENDIF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TeXIncludeGraphics
*!* Author: CULLY Technologies, LLC
*!* Date: 07/15/06 04:29:00 PM
*!* Copyright:
*!* Description: The scale parameter can be "width=1in" or "height=1.5in"
*!*		or "scale=.25", and combinations similar.
*!* Example: THIS.TeXIncludeGraphics([PrunusLaurocerasus1.jpg], [width=1.5in])
*!* Revision Information:
FUNCTION TexIncludeGraphics(tcGraphic AS String, tcScale AS String) AS String
    LOCAL lcRetVal
    lcRetVal = []
	lcGraphic = NVL(tcGraphic, [])
	lcScale = NVL(tcScale, [])
	IF FILE(tcGraphic)
		THIS.TeXAddPackage("graphicx")
		lcRetVal = lcRetVal + [\includegraphics]
		IF NOT EMPTY(lcScale)
			lcRetVal = lcRetVal + "[" + lcScale + "]"
		ENDIF
		lcRetVal = lcRetVal + [{] + lcGraphic + [}] + THIS.CRLF
	ENDIF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TeXIncludePDF
*!* Author: CULLY Technologies, LLC
*!* Date: 07/19/06 10:45:32 AM
*!* Copyright:
*!* Description: Uses the package PDFPages.  Can be used
*!*		to embed a PDF file into the TeX output.  Can
*!*	be used to combine multiple PDFs into one document.
*!*
*!* Revision Information:
FUNCTION TeXIncludePDF(tcPDFFileName AS String) AS String
    LOCAL lcRetVal
    THIS.TeXAddPackage("pdfpages")
    lcRetVal = []
    lcRetVal = lcRetVal + "\includepdf[pages=-]{" + tcPDFFileName + "}" + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*///////////////////////////////////////////////////////
FUNCTION TexBackMatter()
	LOCAL lcRetVal
	lcRetVal = []
	lcRetVal = lcRetVal + THIS.TexBibliography()
	lcRetVal = lcRetVal + THIS.TexIndexPrint()
	lcRetVal = lcRetVal + THIS.TexEnd()
RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexBibliography
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 08:54:41 AM
*!* Copyright:
*!* Description: Creates the bibliography content
*!* Revision Information:
FUNCTION TexBibliography() AS String
	LOCAL lcRetVal, lnOrigArea, loBib
	lcRetVal = []
	lnOrigArea = SELECT()
	IF USED(THIS.cBiblioCursor)
		SELECT (THIS.cBiblioCursor)
		SCATTER NAME loBib
		IF NOT EMPTY(THIS.cBiblioField) AND NOT EMPTY(FIELD(THIS.cBiblioField))
			lcRetVal = lcRetVal + [\begin{thebibliography}{00}] + THIS.CRLF
			SCAN
				SCATTER NAME loBib
				lcBibEntry = EVALUATE([loBib.] + THIS.cBiblioField)
				lcRetVal = lcRetVal + [\bibitem{] + TRANSFORM(RECNO()) + [}] + lcBibEntry + THIS.CRLF
			ENDSCAN
			lcRetVal = lcRetVal + [\end{thebibliography}] + THIS.CRLF + THIS.CRLF
		ENDIF
	ENDIF
	SELECT (lnOrigArea)
RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexIndexMake
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 10:55:10 AM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexIndexMake
	LOCAL lcRetVal
	lcRetVal = []
	THIS.TeXAddPackage("makeidx")
	IF THIS.lShowIndex
		lcRetVal = lcRetVal + [\makeindex] + THIS.CRLF
	ENDIF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexIndexPrint
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/06 10:55:10 AM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TexIndexPrint
	LOCAL lcRetVal
	lcRetVal = []
	THIS.TeXAddPackage("makeidx")
	IF THIS.lShowIndex
		lcRetVal = lcRetVal + [\printindex] + THIS.CRLF
	ENDIF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexIndexItem
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 08:48:12 AM
*!* Copyright:
*!* Description: This takes an string passed to it an marks it to be indexed.
*!* Revision Information:
FUNCTION TexIndexItem(tcItem AS String) AS String
    LOCAL lcRetVal
    lcRetVal = []
	THIS.TeXAddPackage("makeidx")
    IF PCOUNT() = 1 AND NOT ISNULL(tcItem)
	    lcRetVal = lcRetVal + [ \index{] + ALLTRIM(tcItem) + [} ] + THIS.CRLF
    ENDIF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexReference
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/2006
*!* Copyright:
*!* Description: Will create a reference from within text
*!*		to an image or table.  Just pass the same key
*!*		value used within the image or table.
*!* Revision Information:
FUNCTION TexReference(tcKey AS String) AS String
	LOCAL lcRetVal
	lcRetVal = []
	THIS.TeXAddPackage("varioref")
	lcRetVal = lcRetVal + [\vref{] + NVL(tcKey,[]) + [}] + THIS.CRLF
	RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexReferencePage
*!* Author: CULLY Technologies, LLC
*!* Date: 05/13/2006
*!* Copyright:
*!* Description: Will create a reference from within text
*!*		to an image or table.  Just pass the same key
*!*		value used within the image or table.
*!* Revision Information:
FUNCTION TexReferencePage(tcKey AS String) AS String
	LOCAL lcRetVal
	lcRetVal = []
	THIS.TeXAddPackage("varioref")
	lcRetVal = lcRetVal + [\ref{] + NVL(tcKey,[]) + [} on page~\pageref{] + NVL(tcKey,[]) + [}] + THIS.CRLF
	RETURN lcRetVal
ENDFUNC


*!************************************************
*!* Program: TexEnd
*!* Author: CULLY Technologies, LLC
*!* Date: 05/25/06 04:58:29 PM
*!* Copyright:
*!* Description: Marks the end of the TeX document
*!* Revision Information:
FUNCTION TexEnd()
	RETURN [\end{document}] + THIS.CRLF
ENDFUNC

*!************************************************
*!* Program: TexEscChars
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 04:59:04 PM
*!* Copyright:
*!* Description: Makes sure that all special characters in a string are 
*!*		escaped properly.  To make a "^" appear in the end product, it 
*!*		must be escaped properly.
*!* Revision Information:
*!*		6/25/06 KJC - Removed the curly brackets checking.  I need to handle that
*!*			in the future.  The curly brackets are used in \index{} markups.
FUNCTION TexEscChars(tcString AS String) AS String
    LOCAL lcRetVal, lcSpecialCharacters, lnX, lcSpecialChar
    lcRetVal = THIS.TexEscCharsBackslash(tcString)			&& We handle the special case of backslashes
    lcSpecialCharacters = [#$%&~_^]
    FOR lnX = 1 TO LEN(lcSpecialCharacters)
    	lcSpecialChar = SUBSTR(lcSpecialCharacters, lnX, 1)
    	lnLocation = AT(lcSpecialChar, tcString)
    	IF lnLocation > 0									&& Comparing into the first string
    		lcRetVal = STRTRAN(lcRetVal, lcSpecialChar, [\] + lcSpecialChar)
    	ENDIF
    ENDFOR
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexEscCharsBackslash
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 05:23:41 PM
*!* Copyright:
*!* Description: This handles the special case of searching for backslashes
*!*		in text.  We have to be especially careful because certain data may 
*!*		already have TeX tags within it.
*!*
*!*		I structured this function with a CASE statement to allow for additional
*!*		Tex markups in the future in addition to \index which I'm handling now.
*!* Revision Information:
FUNCTION TexEscCharsBackslash(tcString AS String) AS String
    LOCAL lcRetVal, lnLocation, lnCloseLocation
    lcRetVal = []
    lnLocation = AT([\], tcString)
    IF lnLocation = 0
    	lcRetVal = tcString
    ELSE
    	lcRetVal = LEFT(tcString, lnLocation)			&& We've included the slash
    	tcString = SUBSTR(tcString, lnLocation + 1)		&& We've chopped off the string just after the slash
    	DO CASE
    		CASE LEFT(LOWER(tcString), 5) = [index{]
    			lnCloseLocation = AT([}], tcString)
    			IF lnCloseLocation = 0
    				SET STEP ON
    			ELSE
	    			lcRetVal = lcRetVal + LEFT(tcString, lnCloseLocation)
	    			tcString = SUBSTR(tcString, lnCloseLocation + 1)
    			ENDIF
    		OTHERWISE
    	ENDCASE
    	lcRetVal = lcRetVal + THIS.TexEscCharsBackslash(tcString)
    ENDIF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TeXEquation
*!* Author: CULLY Technologies, LLC
*!* Date: 07/15/06 08:28:15 AM
*!* Copyright:
*!* Description: Do not pass any unintended blank spaces.  TeX will interpret them as markers for new paragraph
*!*		and will generate an error.
*!* Example: THIS.TeXEquation([\varph(x,z) = z - \gamma_{10} x - \sum_{m+n\ge2} \gamma{mn} x^m z^n])
*!* Revision Information:
FUNCTION TeXEquation(tcEquation AS String) AS String
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\begin{equation}] + THIS.CRLF
    lcRetVal = lcRetVal + tcEquation + THIS.CRLF
    lcRetVal = lcRetVal + [\end{equation}]	+ THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TexDisplayMath
*!* Author: CULLY Technologies, LLC
*!* Date: 07/15/06 08:20:54 AM
*!* Copyright:
*!* Description:  Used for the display of larger formulas and equations.
*!* Example: THIS.TeXDisplayMath([])
*!* Revision Information:
FUNCTION TexDisplayMath(tcEquation AS String) AS String
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\begin{displaymath}] + THIS.CRLF
    lcRetVal = lcRetVal + tcEquation + THIS.CRLF
    lcRetVal = lcRetVal + [\end{displaymath}]	+ THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: Version
*!* Author: CULLY Technologies, LLC
*!* Date: 07/19/06 09:38:28 AM
*!* Copyright:
*!* Description: Returns the version of this class.
*!* Revision Information:
FUNCTION VERSION() AS String
    LOCAL lcRetVal
    lcRetVal = [1.1]
    RETURN lcRetVal
ENDFUNC

ENDDEFINE
*///////////////////////////////////////////////////////
*///////////////////////////////////////////////////////


*///////////////////////////////////////////////////////
*///////////////////////////////////////////////////////
*///////////////////////////////////////////////////////
*!* Description:
*!* 	This class is meant to help in creating TeX tables.
*!*	Note that a TeX table does not constitute a full TeX
*!*	document and needs to be wrapped with appropriate tags.
DEFINE CLASS CursorToTable AS Custom

	CRLF				= []
	cTable				= []				&& Actually holds the output
	cTabularBanner		= []				&& Holds text that spans the top of the tabular.
	cTableCaption 		= []
	cSourceCursor		= []				&& Cursor name
	nSourceFieldCount 	= 0					&& Holds the number of fields (columns) in the source cursor
	oTeX				= .NULL.			&& TeX object
	
*!************************************************
*!* Program: Init
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 09:01:16 AM
*!* Copyright:
*!* Description: Takes care of setting up our object for use.
*!* Revision Information:	
FUNCTION Init
	THIS.CRLF = CHR(10)
	THIS.oTex = CREATEOBJECT("FoxToTeX")
	DODEFAULT()
ENDFUNC

*!************************************************
*!* Program: ResetProperties
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 09:56:57 AM
*!* Copyright:
*!* Description: This takes the properties for this class and resets them back to empty.
*!* Revision Information:
FUNCTION ResetProperties
	STORE [] TO THIS.cTable, THIS.cSourceCursor
ENDFUNC

*!************************************************
*!* Program: GenerateTable
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 09:58:43 AM
*!* Copyright:
*!* Description: Takes care of generically generating a TeX table.
*!* Revision Information:
FUNCTION GenerateTable() AS String
	LOCAL lcRetVal
	lcRetVal = []
	IF NOT EMPTY(THIS.cSourceCursor)
		IF USED(THIS.cSourceCursor)
			THIS.SourceFieldCount()
			lcRetVal = lcRetVal + THIS.CRLF
			THIS.oTex.TexAddPackage('longtable')
			lcRetVal = lcRetVal + THIS.TableBegin()
			lcRetVal = lcRetVal + THIS.TableBeginAlign([center])
			lcRetVal = lcRetVal + THIS.TabularBegin()
			lcRetVal = lcRetVal + THIS.TabularBanner()
			lcRetVal = lcRetVal + THIS.TabularHeaders()
			lcRetVal = lcRetVal + THIS.TabularBody()
			lcRetVal = lcRetVal + THIS.TabularEnd()

			IF NOT ISNULL(THIS.cTableCaption) AND NOT EMPTY(THIS.cTableCaption)
				lcRetVal = lcRetVal + THIS.TableCaption(THIS.cTableCaption)
			ENDIF
			lcRetVal = lcRetVal + THIS.TableEndAlign([center])
			lcRetVal = lcRetVal + THIS.TableEnd()
			lcRetVal = lcRetVal + THIS.CRLF
		ENDIF
	ENDIF
    RETURN lcRetVal
ENDFUNC


*!************************************************
*!* Program: TableBegin
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 04:35:50 PM
*!* Copyright:
*!* Description: Creates the table begin tag
*!* Revision Information:
FUNCTION TableBegin
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\begin{longtable}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TableEnd
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 04:36:48 PM
*!* Copyright:
*!* Description: Creates the table end tag
*!* Revision Information:
FUNCTION TableEnd
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\end{longtable}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TableBeginAlign
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 04:35:50 PM
*!* Copyright:
*!* Description: Creates the table begin alignment tag
*!* Revision Information:
FUNCTION TableBeginAlign(tcMode AS String)
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\begin{]+tcMode+[}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TableEnd
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 04:36:48 PM
*!* Copyright:
*!* Description: Creates the table end alignment tag
*!* Revision Information:
FUNCTION TableEndAlign(tcMode AS String)
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\end{]+tcMode+[}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC


*!************************************************
*!* Program: SourceFieldCount
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:24:30 AM
*!* Copyright:
*!* Description: Returns the number of fields (columns) in the source cursor.
*!* Revision Information:
FUNCTION SourceFieldCount() AS Integer
	LOCAL lnRetVal
	THIS.nSourceFieldCount = FCOUNT(THIS.cSourceCursor)
    RETURN THIS.nSourceFieldCount
ENDFUNC

*!************************************************
*!* Program: ColumnAlignment
*!* Author: CULLY Technologies, LLC
*!* Date: 07/16/06 07:28:32 PM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION ColumnAlignment
	LOCAL lnX, lcField, lcType, lcRetVal, loCursor
	lcRetVal = [{]
	SELECT (THIS.cSourceCursor)
	SCATTER NAME loCursor MEMO BLANK
	FOR lnX = 1 TO THIS.nSourceFieldCount
		lcField = FIELD(lnX)
		lcType = TYPE(THIS.cSourceCursor + [.] + lcField)
		lcRetVal = lcRetVal + [|]
		DO CASE
			CASE INLIST(lcType, "N")		&& right aligned
				lcRetVal = lcRetVal + [r]
			CASE INLIST(lcType, "L")		&& right aligned
				lcRetVal = lcRetVal + [c]
			CASE INLIST(lcType, "M")		&& fixed width, 2 inches wide
				lcRetVal = lcRetVal + [p{4in}]
			OTHERWISE						&& left aligned
				lcRetVal = lcRetVal + [l]
		ENDCASE
	ENDFOR
	lcRetVal = lcRetVal + [|}]				&& Close the field alignment tag
RETURN lcRetVal
ENDFUNC


*!************************************************
*!* Program: TableHLine
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:06:15 AM
*!* Copyright:
*!* Description: Returns a "hline" markup
*!* Revision Information:
FUNCTION TableHLine() AS String
	LOCAL lcRetVal
	lcRetVal = [\hline] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TabularBegin
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:04:47 AM
*!* Copyright:
*!* Description: Returns a TeX Tabular markup that indicates the beginning of a Tabular
*!* Revision Information:
FUNCTION TabularBegin() AS String
	LOCAL lcRetVal, loCursor, lnX, lcField, lcType
	lcRetVal = []
	lcRetVal = lcRetVal + [\begin{tabular}]
	lcRetVal = lcRetVal + THIS.ColumnAlignment()
	lcRetVal = lcRetVal + THIS.CRLF
	lcRetVal = lcRetVal + THIS.TableHLine()
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TabularBanner
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:17:05 AM
*!* Copyright:
*!* Description: Takes care of printing a Tabular banner if the property is non blank
*!* Revision Information:
FUNCTION TabularBanner() AS String
	LOCAL lcRetVal
	lcRetVal = []
	IF NOT EMPTY(THIS.cTabularBanner)
		lcRetVal = lcRetVal + [\multicolumn{]
		lcRetVal = lcRetVal + TRANSFORM(THIS.nSourceFieldCount)
		lcRetVal = lcRetVal + [}]
		lcRetVal = lcRetVal + [{|c|}]
		lcRetVal = lcRetVal + [{] + THIS.cTabularBanner + [}\\] + THIS.CRLF
		lcRetVal = lcRetVal + THIS.TableHLine()
	ENDIF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TableCaption
*!* Author: CULLY Technologies, LLC
*!* Date: 06/25/06 09:24:04 AM
*!* Copyright:
*!* Description:
*!* Revision Information:
FUNCTION TableCaption(tcCaption AS String) AS String
    LOCAL lcRetVal
    lcRetVal = []
    IF PCOUNT() = 1
    	IF NOT ISNULL(tcCaption)
    		IF NOT EMPTY(tcCaption)
    			IF VARTYPE(tcCaption) = "C"
	    			lcRetVal = lcRetVal + [\caption{]+ALLTRIM(tcCaption)+[}] + THIS.CRLF
    			ENDIF
    		ENDIF
    	ENDIF
    ENDIF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TabularHeaders
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:04:47 AM
*!* Copyright:
*!* Description: Returns a TeX Tabular markup that lists the column headers.
*!* Revision Information:
FUNCTION TabularHeaders() AS String
	LOCAL lcRetVal, loCursor, lnX, lcField
	lcRetVal = []
	SELECT (THIS.cSourceCursor)
	SCATTER NAME loCursor BLANK
	FOR lnX = 1 TO THIS.nSourceFieldCount
		lcField = FIELD(lnX,THIS.cSourceCursor)
		lcField = IIF(EMPTY(lcField), FIELD(lnX), lcField)
		lcRetVal = lcRetVal + IIF(EMPTY(lcRetVal), [], [ & ])
		lcRetVal = lcRetVal + lcField
	ENDFOR
	lcRetVal = lcRetVal + [\\]				&& Close the column tag
	lcRetVal = lcRetVal + THIS.CRLF
	lcRetVal = lcRetVal + THIS.TableHLine()
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TabularBody
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:48:47 AM
*!* Copyright:
*!* Description: Returns a TeX Tabular markup that lists the rows and column values from the source cursor.
*!* Revision Information:
FUNCTION TabularBody() AS String
	LOCAL lcRetVal, loRow
	lcRetVal = []
	SELECT (THIS.cSourceCursor)
	SCAN
		SCATTER NAME loRow MEMO
		lcRetVal = lcRetVal + THIS.TabularBodyRow(loRow)
		lcRetVal = lcRetVal + [\\]				&& Close the column tag
		lcRetVal = lcRetVal + THIS.CRLF
	ENDSCAN
	lcRetVal = lcRetVal + THIS.TableHLine()
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: TabularBodyRow
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:51:55 AM
*!* Copyright:
*!* Description: Creates the Tabular row from an object passed to it.
*!* Revision Information:
FUNCTION TabularBodyRow(toRow AS Object) AS String
    LOCAL lcRetVal, lnX, lcField, luValue, lcValue
    lcRetVal = []
    FOR lnX = 1 TO THIS.nSourceFieldCount
    	lcField = FIELD(lnX)
    	luValue = EVALUATE([toRow.] + lcField)
    	lcValue = THIS.oTeX.TexEscChars(TRANSFORM(luValue))
    	lcRetVal = lcRetVal + IIF(EMPTY(lcRetVal), [], [ & ])
    	lcRetVal = lcRetVal + ALLTRIM(lcValue)
    ENDFOR
    
    RETURN lcRetVal
ENDFUNC


*!************************************************
*!* Program: TabularEnd
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:30:32 AM
*!* Copyright:
*!* Description: Returns the closing tags for a TeX Tabular.
*!* Revision Information:
FUNCTION TabularEnd
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\end{tabular}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC


*!************************************************
*!* Program: GenerateLongTable
*!* Author: CULLY Technologies, LLC
*!* Date: 07/16/06 09:58:43 AM
*!* Copyright:
*!* Description: Takes care of generically generating a TeX table.
*!* Revision Information:
FUNCTION GenerateLongTable() AS String
	LOCAL lcRetVal
	lcRetVal = []
	IF NOT EMPTY(THIS.cSourceCursor)
		IF USED(THIS.cSourceCursor)
			THIS.SourceFieldCount()
			lcRetVal = lcRetVal + THIS.CRLF
			THIS.oTex.TexAddPackage('longtable')
			lcRetVal = lcRetVal + THIS.LongTableBegin()
			lcRetVal = lcRetVal + THIS.LongTableHeaders()
			lcRetVal = lcRetVal + THIS.LongTableBody()
			lcRetVal = lcRetVal + THIS.LongTableEnd()
			IF NOT ISNULL(THIS.cTableCaption) AND NOT EMPTY(THIS.cTableCaption)
				lcRetVal = lcRetVal + THIS.TableCaption(THIS.cTableCaption)
			ENDIF
			lcRetVal = lcRetVal + THIS.CRLF
		ENDIF
	ENDIF
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: LongTableBegin
*!* Author: CULLY Technologies, LLC
*!* Date: 07/16/06 10:04:47 AM
*!* Copyright:
*!* Description: Returns a TeX longtable markup that indicates the beginning of a Tabular
*!* Revision Information:
FUNCTION LongTableBegin() AS String
	LOCAL lcRetVal, loCursor, lnX, lcField, lcType
	lcRetVal = []
	lcRetVal = lcRetVal + [\begin{longtable}]
	lcRetVal = lcRetVal + THIS.ColumnAlignment()
	lcRetVal = lcRetVal + THIS.CRLF
	lcRetVal = lcRetVal + THIS.TableHLine()
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: LongTableHeaders
*!* Author: CULLY Technologies, LLC
*!* Date: 07/16/06 10:04:47 AM
*!* Copyright:
*!* Description: Returns a TeX Tabular markup that lists the column headers.
*!* Revision Information:
FUNCTION LongTableHeaders() AS String
	LOCAL lcRetVal, loCursor, lnX, lcField
	lcRetVal = []
	SELECT (THIS.cSourceCursor)
	SCATTER NAME loCursor BLANK
	FOR lnX = 1 TO THIS.nSourceFieldCount
		lcField = FIELD(lnX,THIS.cSourceCursor)
		lcField = IIF(EMPTY(lcField), FIELD(lnX), lcField)
		lcRetVal = lcRetVal + IIF(EMPTY(lcRetVal), [], [ & ])
		lcRetVal = lcRetVal + [{\em ] + lcField + [}]
	ENDFOR
	lcRetVal = lcRetVal + [\\]				&& Close the column tag
	lcRetVal = lcRetVal + THIS.CRLF
	lcRetVal = lcRetVal + THIS.TableHLine()
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: LongTableBody
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:48:47 AM
*!* Copyright:
*!* Description: Returns a TeX Tabular markup that lists the rows and column values from the source cursor.
*!* Revision Information:
FUNCTION LongTableBody() AS String
	LOCAL lcRetVal, loRow
	lcRetVal = []
	SELECT (THIS.cSourceCursor)
	SCAN
		SCATTER NAME loRow MEMO
		lcRetVal = lcRetVal + THIS.LongTableBodyRow(loRow)
		lcRetVal = lcRetVal + [\\]				&& Close the column tag
		lcRetVal = lcRetVal + THIS.CRLF
	ENDSCAN
	lcRetVal = lcRetVal + THIS.TableHLine()
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: LongTableBodyRow
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:51:55 AM
*!* Copyright:
*!* Description: Creates the Tabular row from an object passed to it.
*!* Revision Information:
FUNCTION LongTableBodyRow(toRow AS Object) AS String
    LOCAL lcRetVal, lnX, lcField, luValue, lcValue
    lcRetVal = []
    FOR lnX = 1 TO THIS.nSourceFieldCount
    	lcField = FIELD(lnX)
    	luValue = EVALUATE([toRow.] + lcField)
    	lcValue = THIS.oTeX.TexEscChars(TRANSFORM(luValue))
    	lcRetVal = lcRetVal + IIF(EMPTY(lcRetVal), [], [ & ])
    	lcRetVal = lcRetVal + ALLTRIM(lcValue)
    ENDFOR
    
    RETURN lcRetVal
ENDFUNC

*!************************************************
*!* Program: LongTableEnd
*!* Author: CULLY Technologies, LLC
*!* Date: 06/20/06 10:30:32 AM
*!* Copyright:
*!* Description: Returns the closing tags for a TeX Tabular.
*!* Revision Information:
FUNCTION LongTableEnd
    LOCAL lcRetVal
    lcRetVal = []
    lcRetVal = lcRetVal + [\end{longtable}] + THIS.CRLF
    RETURN lcRetVal
ENDFUNC




ENDDEFINE
*///////////////////////////////////////////////////////
*///////////////////////////////////////////////////////
